/*
 * QrfeTraceModule.h
 *
 *  Created on: 31.01.2011
 *      Author: stefan.detter
 */

#ifndef QRFETRACEMODULE_H_
#define QRFETRACEMODULE_H_

#include <QString>
#include <QStringList>


class QrfeTraceModule;

class QrfeTraceFunction
{
public:
	QrfeTraceFunction ( const QrfeTraceModule* module, const QString& name, QStringList args = QStringList() );
	~QrfeTraceFunction ( );

private:
	const QrfeTraceModule* 	m_module;
    const QString 			m_name;
};

class QrfeTraceModule
{
public:
	/*!
	 * @brief Constructor of the TraceModule
	 * By subclassing this class the trace functions can be used very easy even with the module name extension.
	 */
	QrfeTraceModule(QString name);

	/*!
	 * @brief Constructor of the TraceModule
	 * By subclassing this class the trace functions can be used very easy even with the module name extension.
	 */
	QrfeTraceModule(QString name, uint minimumTraceLevel);

	/*!
	 * @brief Constructor of the TraceModule
	 * By subclassing this class the trace functions can be used very easy even with the module name extension.
	 */
	QrfeTraceModule(QString name, uint minimumTraceLevel, uint functionTraceLevel);

	/*!
	 * @brief Standard destructor
	 */
	virtual ~QrfeTraceModule();

protected:

	/*!
	 * @brief Function to trace a warning.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr.
	 * @param	msg		Trace message
	 */
	virtual void warning(const QString& msg) const;

	/*!
	 * @brief Function to trace an error.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr.
	 * @param	msg		Trace message
	 */
	virtual void error(const QString& msg) const;

	/*!
	 * @brief Function to trace a fatal error.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr. After this the application is closed.
	 * @param	msg		Trace message
	 */
	virtual void fatal(const QString& msg) const;

	/*!
	 * @brief Function to trace a message.
	 * The time, module and thread info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace.
	 * @param 	trcLevel	The trace level of the message
	 * @param	msg			Trace message
	 */
	virtual void trc(const int trcLevel, const QString& msg) const;

	/*!
	 * @brief Function to trace a message with additional bytes.
	 * The time, module and thread info are prepended to the given message. The given bytes are appended to the message
	 * in the given pattern. After this the message is traced out with the QrfeTrace.
	 * @param 	trcLevel	The trace level of the message
	 * @param	msg			Trace message
	 * @param	bytes		The additional bytes
	 * @param	pattern		The pattern how the bytes should be written
	 */
	virtual void trcBytes(const int trcLevel, const QString& msg,
			const QByteArray& bytes, const QString& pattern = "0x%1 ") const;

public:
	/*!
	 * @brief Static function to trace a warning.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr.
	 * @param	msg		Trace message
	 */
	static void swarning(const QString& msg);

	/*!
	 * @brief Static function to trace an error.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr.
	 * @param	msg		Trace message
	 */
	static void serror(const QString& msg);

	/*!
	 * @brief Static function to trace a fatal error.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr. After this the application is closed.
	 * @param	msg		Trace message
	 */
	static void sfatal(const QString& msg);

	/*!
	 * @brief Static function to trace a message.
	 * The time, module and thread info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace.
	 * @param 	trcLevel	The trace level of the message
	 * @param	msg			Trace message
	 */
	static void strc(const int trcLevel, const QString& msg);

	/*!
	 * @brief Static function to trace a message with additional bytes.
	 * The time, module and thread info are prepended to the given message. The given bytes are appended to the message
	 * in the given pattern. After this the message is traced out with the QrfeTrace.
	 * @param 	trcLevel	The trace level of the message
	 * @param	msg			Trace message
	 * @param	bytes		The additional bytes
	 * @param	pattern		The pattern how the bytes should be written
	 */
	static void strcBytes(const int trcLevel, const QString& msg,
			const QByteArray& bytes, const QString& pattern = "0x%1 ");


public:
	/*!
	 * @brief Static function to trace a warning.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr.
	 * @param	moduleName		Name of the tracing module
	 * @param	msg				Trace message
	 */
	static void swarning(const QString& moduleName, const QString& msg);

	/*!
	 * @brief Static function to trace an error.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr.
	 * @param	moduleName	Name of the tracing module
	 * @param	msg			Trace message
	 */
	static void serror(const QString& moduleName, const QString& msg);

	/*!
	 * @brief Static function to trace a fatal error.
	 * The time, module, thread and type info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace and additional written to stderr. After this the application is closed.
	 * @param	moduleName	Name of the tracing module
	 * @param	msg			Trace message
	 */
	static void sfatal(const QString& moduleName, const QString& msg);

	/*!
	 * @brief Static function to trace a message.
	 * The time, module and thread info are prepended to the given message. After this the message is traced out
	 * with the QrfeTrace.
	 * @param	moduleName	Name of the tracing module
	 * @param 	trcLevel	The trace level of the message
	 * @param	msg			Trace message
	 */
	static void strc(const QString& moduleName, const int trcLevel,
			const QString& msg);

	/*!
	 * @brief Static function to trace a message with additional bytes.
	 * The time, module and thread info are prepended to the given message. The given bytes are appended to the message
	 * in the given pattern. After this the message is traced out with the QrfeTrace.
	 * @param	moduleName	Name of the tracing module
	 * @param 	trcLevel	The trace level of the message
	 * @param	msg			Trace message
	 * @param	bytes		The additional bytes
	 * @param	pattern		The pattern how the bytes should be written
	 */
	static void strcBytes(const QString& moduleName, const int trcLevel,
			const QString& msg, const QByteArray& bytes, const QString& pattern =
					"0x%1 ");
protected:
	/*!
	 * @brief Simple helper function to create the module name with the right count of spaces.
	 */
	static QString prepareModuleName(const QString& moduleName);


protected:
	void setMinimumTraceLevel ( uint minimumTraceLevel );

	void setFunctionTraceLevel ( uint functionTraceLevel );

protected:
	QString 	traceModuleName;
	uint 		minimumTraceLevel;
	uint 		functionTraceLevel;

	friend class QrfeTraceFunction;
};

#endif /* QRFETRACEMODULE_H_ */
